#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <SDL/SDL.h>
#include "couche.h"
#include "interface.h"

void cart_pol(double *cart, double *pol)
{
    double t;

    // Recherche de l'argument
    if(cart[0] == 0.0)
    {
        if(cart[1] < 0.0) {t = -M_PI / 2.0;}
        else {t = M_PI / 2.0;}
    }
    else {t = atan(cart[1] / cart[0]);}
    if(cart[0] < 0.0) {t = t + M_PI;}
    if(t < 0.0) {t += M_PI * 2.0;} // Argument positif (dans [0;2PI])

    pol[0] = sqrt(cart[0] * cart[0] + cart[1] * cart[1]); // Calcul du module
    pol[1] = t; // Argument
    return;
}

void normalisation(double **exemple, double nbr_exemples, double nbr_entrees, t_coord coord)
{
    double *min, *max;
    long i, j;

    // Initialisation
    min = (double *)malloc(nbr_entrees * sizeof(double));
    if(!min) {exit(EXIT_FAILURE);}
    max = (double *)malloc(nbr_entrees * sizeof(double));
    if(!max) {exit(EXIT_FAILURE);}
    for(j = 0; j < nbr_entrees; j++)
    {
        min[j] = exemple[0][j];
        max[j] = exemple[0][j];
    }

    // Recherche des bornes
    for(i = 1; i < nbr_exemples; i++)
    {
        for(j = 0; j < nbr_entrees; j++)
        {
            if(min[j] > exemple[i][j]) {min[j] = exemple[i][j];}
            if(max[j] < exemple[i][j]) {max[j] = exemple[i][j];}
        }
    }

    // Normalisation particulire
    if(coord == CART) // Si coordonnes catrsiennes : on conserve l'orthonormalit
    {
        if(min[0] < min[1]) {min[1] = min[0];}
        else {min[0] = min[1];}
        if(max[0] > max[1]) {max[1] = max[0];}
        else {max[0] = max[1];}
    }
    else if(coord == POL) // Si coordonnes polaires : on a des simplifications possibles
    {
        min[0] = 0.0;
        min[1] = 0.0;
        max[1] = 2.0 * M_PI;
    }

    for(i = 0; i < nbr_exemples; i++) {for(j = 0; j < nbr_entrees; j++){exemple[i][j] = (exemple[i][j] - min[j]) / (max[j] - min[j]);}} // Normalisation
    free(min);
    free(max);
    return;
}

long nouveau_reseau(t_reseau *res, long nbr_couches, long *nbr_neurones, double interval_poids)
{
    long i, j, k; // Compteurs

    /// Initialisation de la description du rseau
    res->nbr_couches = nbr_couches;
    res->nbr_neurones = (long *)malloc(nbr_couches * sizeof(long));
    if(!res->nbr_neurones) {exit(EXIT_FAILURE);}
    res->resultat = (double **)malloc(nbr_couches * sizeof(double *));
    if(!res->resultat) {exit(EXIT_FAILURE);}
    res->delta = (double **)malloc(nbr_couches * sizeof(double *));
    if(!res->delta) {exit(EXIT_FAILURE);}
    res->poids = (double ***)malloc(nbr_couches * sizeof(double **));
    if(!res->poids) {exit(EXIT_FAILURE);}

    /// Initialisation de la description de chaque couche
    res->nbr_neurones[0] = nbr_neurones[0];
    res->resultat[0] = NULL;
    res->delta[0] = NULL;
    res->poids[0] = NULL;
    for(i = 1; i < nbr_couches - 1; i++) // Pour chaque couche (sauf celle d'entre et de sortie)
    {
        res->resultat[i] = (double *)malloc(nbr_neurones[i] * sizeof(double));
        if(!res->resultat[i]) {exit(EXIT_FAILURE);}
    }
    for(i = 1; i < nbr_couches; i++) // Pour chaque couche (sauf celle d'entre)
    {
        /// Initialisation des neurones de la couche
        res->nbr_neurones[i] = nbr_neurones[i];
        res->delta[i] = (double *)malloc(nbr_neurones[i] * sizeof(double));
        if(!res->delta[i]) {exit(EXIT_FAILURE);}
        res->poids[i] = (double **)malloc(nbr_neurones[i] * sizeof(double *));
        if(!res->poids[i]) {exit(EXIT_FAILURE);}

        for(j = 0; j < nbr_neurones[i]; j++) // Pour chaque neurone de la couche
        {
            /// Initialisation des synapses
            res->poids[i][j] = (double *)malloc(nbr_neurones[i - 1] * sizeof(double));
            if(!res->poids[i][j]) {exit(EXIT_FAILURE);}
            for(k = 0; k < nbr_neurones[i - 1]; k++) {res->poids[i][j][k] = (((double)rand() / RAND_MAX - 0.5) * interval_poids);} // Initialisation alatoire de chaque poids
            /// Fin de l'initialisation des synapses
        }
        /// Fin de l'initialisation des neurones de la couche
    }
    /// Fin de l'initialisation des couches

    res->forme = (SDL_Color *)malloc(res->nbr_neurones[res->nbr_couches - 1] * sizeof(SDL_Color));
    if(!res->forme) {exit(EXIT_FAILURE);}
    res->temps = (long *)malloc(sizeof(long));
    if(!res->temps) {exit(EXIT_FAILURE);}
    *(res->temps) = 0;
    /// Fin de l'initialisation du rseau
    return 1;
}

void supprimer_reseau(t_reseau res)
{
    long i, j; // Compteur

    supprimer_exemples(res); // On supprime les exemples du rseau
    for(i = 1; i < res.nbr_couches - 1; i++) {free(res.resultat[i]);} // Pour chaque couche (sauf celle d'entre et de sortie)
    for(i = 1; i < res.nbr_couches; i++) // Pour chaque couche (sauf celle d'entre)
    {
        for(j = 0; j < res.nbr_neurones[i]; j++) {free(res.poids[i][j]);} // Pour tous les neurones de la couche
        free(res.poids[i]);
        free(res.delta[i]);
    }
    free(res.temps);
    free(res.forme);
    free(res.poids);
    free(res.delta);
    free(res.resultat);
    free(res.nbr_neurones);
    return;
}

long generer_spirales(t_reseau *res, long nbr_pts, long offset, t_coord coord)
{
    double t; // Valeur de t courante (cf. sujet)
    long i; // Compteur qui parcourt les points

    /// Initialisation de la description de l'ensemble d'exemple
    res->mode = SPIRALES; // On va dessiner les 2 spirales
    res->offset = offset; // Nombres de points ignors
    res->coord = coord; // Systme de coordonnes utilises
    res->nbr_pts = nbr_pts; // Nombre de points dans l'ensemble d'exemple

    /// Allocation mmoire pour l'ensemble d'exemple
    res->pts = (double **)malloc(nbr_pts * sizeof(double *));
    if(!res->pts) {exit(EXIT_FAILURE);}
    res->att = (double **)malloc(nbr_pts * sizeof(double *));
    if(!res->att) {exit(EXIT_FAILURE);}
    for(i = 0; i < nbr_pts; i++) // Pour chaque point  apprendre
    {
        res->pts[i] = (double *)malloc(res->nbr_neurones[0] * sizeof(double)); // Il y a autant d'entres que de neurones d'entres
        if(!res->pts[i]) {exit(EXIT_FAILURE);}
        res->att[i] = (double *)malloc(res->nbr_neurones[res->nbr_couches - 1] * sizeof(double)); // Il y a autant de sorties que de neurones de sorties
        if(!res->att[i]) {exit(EXIT_FAILURE);}
    }

    /// Remplissage en coordonnes cartsienne (d'abord)
    t = SPIRALES_PAS * (offset + 1); // On ne prend pas en compte les points premiers points (dont l'origine)
    for(i = 0; i < nbr_pts; i++) // Pour chaque point, on apprend alternativement un point de la premire ou de la deuxime spirale
    {
        res->pts[i][0] = t * cos(t + res->phase);
        res->pts[i][1] = t * sin(t + res->phase);
        if(i%2)
        {
            res->pts[i][0] = -res->pts[i][0];
            res->pts[i][1] = -res->pts[i][1];
            res->att[i][0] = 0.0;
            res->att[i][1] = 1.0;
            t += SPIRALES_PAS;
        }
        else
        {
            res->att[i][0] = 1.0;
            res->att[i][1] = 0.0;
        }
    }

    /// Conversion dans le systme de coordonnes dsir
    switch(coord) // En fonction du systme de coordonnes
    {
        case CART: // Il n'y a plus rien  faire
            break;
        case POL: // Conversion
            for(i = 0; i < nbr_pts; i++) {cart_pol(res->pts[i], res->pts[i]);} // On convertit chaque point
            break;
        default: // Systme de coordonne inconnue
            return 0;
            break;
    }

    /// Normalisation
    normalisation(res->pts, nbr_pts, res->nbr_neurones[0], res->coord);
    return 1;
}

void supprimer_exemples(t_reseau res)
{
    long i;

    // Libration de chaque exemple
    for(i = 0; i < res.nbr_pts; i++)
    {
        free(res.pts[i]);
        free(res.att[i]);
    }
    free(res.pts);
    free(res.att);
    return;
}

long sauvegarder_reseau(t_reseau res, char *nom_reseau)
{
    FILE *fichier; // Pointeur sur le fichier dans lequel on enregistre
    char nom_fichier[SAISIE_MAX];
    long i, j; // Compteurs

    sprintf(nom_fichier,"%s%s%s",NEU_DIR,nom_reseau,NEU_EXT);
    fichier = fopen(nom_fichier, "wb"); // On ouvre le fichier en criture
    if(fichier == NULL) {return 0;} // Si on n'a pas russi on arrte

    fwrite("MNGX", 1, 4, fichier); // On marque le fichier pour tre sr qu'il a t gnr par ce programme
    fwrite(&(res.epsilon), sizeof(double), 1, fichier); // On crit epsilon
    fwrite(&(res.seuil), sizeof(double), 1, fichier); // On crit le seuil
    fwrite(&(res.coord), sizeof(t_coord), 1, fichier); // On crit le systeme de coordonnes

    fwrite(&(res.nbr_couches), sizeof(long), 1, fichier); // On crit le nombre de couches
    fwrite(res.nbr_neurones, sizeof(long), res.nbr_couches, fichier); // On crit le nombre de neurones de chaque couche
    for(i = 1; i < res.nbr_couches; i++) // Pour chaque couche (sauf celle d'entre)
    {
        for(j = 0; j < res.nbr_neurones[i]; j++) // Pour chaque neurone de la couche
        {
            fwrite(res.poids[i][j], sizeof(double), res.nbr_neurones[i - 1], fichier); // On crit les poids synaptiques en entre du neurone
        }
    }
    fwrite(res.forme, sizeof(SDL_Color), res.nbr_neurones[res.nbr_couches - 1], fichier); // On crit les couleurs de chaque forme

    fwrite(&(res.mode), sizeof(t_mode), 1, fichier); // On crit le mode
    fwrite(&(res.nbr_pts), sizeof(long), 1, fichier); // On crit le nombre de points
    switch(res.mode)
    {
        case SPIRALES: // Si le rseau analyse des spirales
            fwrite(&(res.offset), sizeof(long), 1, fichier); // On crit l'offset
            fwrite(&(res.phase), sizeof(double), 1, fichier); // On crit la phase
            break;
        case PERSO: // Dans le cas d'un dessin on fait un enregistrement par dfaut
        default:
            for(i = 0; i < res.nbr_pts; i++) // Pour chaque exemple
            {
                fwrite(res.pts[i], sizeof(double), res.nbr_neurones[0], fichier); // On crit toutes les entres de cet exemple
                fwrite(res.att[i], sizeof(double), res.nbr_neurones[res.nbr_couches - 1], fichier); // Et les sorties attendues
            }
            break;
    }
    fwrite(res.temps, sizeof(long), 1, fichier); // On crit le temps d'apprentissage

    fclose(fichier); // On ferme le fichier
    return 1;
}

long charger_reseau(t_reseau *res, char *nom_reseau, long charge, double interval_poids)
{
    FILE *fichier; // Pointeur sur le fichier depuis lequel on charge
    char nom_fichier[SAISIE_MAX];
    char c[] = "MNGX"; // Recoit les caractres de contrle
    long t1, *t2; // Contiendront le nombre de couche et de neurones par couche
    long i, j; // Compteurs

    sprintf(nom_fichier, "%s%s%s", NEU_DIR, nom_reseau, NEU_EXT);
    fichier = fopen(nom_fichier, "rb"); // On ouvre le fichier en lecture
    if(fichier == NULL) {return 0;} // Si on n'a pas russi on arrte

    fread(c, 1, 4, fichier); // On charge les caractres de contrle
    if(strcmp(c, "MNGX")) // On effectue le contrle, si les chane sont diffrentes : on ferme le fichier et on indique que le chargement ne s'est pas effectu
    {
        fclose(fichier);
        return 0;
    }

    if (charge)
    {
        supprimer_reseau(*res);
    }

    fread(&(res->epsilon), sizeof(double), 1, fichier); // On charge epsilon
    fread(&(res->seuil), sizeof(double), 1, fichier); // On charge le seuil
    fread(&(res->coord), sizeof(t_coord), 1, fichier); // On charge le systme de coordones

    fread(&(t1), sizeof(long), 1, fichier); // On charge le nombre de couches
    t2 = (long *)malloc(t1 * sizeof(long)); // On alloue la mmoire ncessaire pour lire les nombres de neurones par couches
    if(t2 == NULL) // Si on n'a pas russi
    {
        fclose(fichier); // On ferme le fichier
        return 0; // On arrte
    }
    fread(t2, sizeof(long), t1, fichier); // On charge les nombres de neurones par couches
    nouveau_reseau(res, t1, t2,interval_poids); // On alloue la mmoire ncessaire pour tout le rseau
    for(i = 1; i < t1; i++) // Pour chaque couche (sauf celle d'entre)
    {
        for(j = 0; j < t2[i]; j++) // Pour chaque neurone de la couche
        {
            fread(res->poids[i][j], sizeof(double), t2[i - 1], fichier); // On charge les poids synaptiques en entre du neurone
        }
    }
    fread(res->forme, sizeof(SDL_Color), res->nbr_neurones[res->nbr_couches - 1], fichier); // On crit les couleurs de chaque forme

    fread(&(res->mode), sizeof(t_mode), 1, fichier); // On charge le mode
    fread(&(res->nbr_pts), sizeof(long), 1, fichier); // On charge le nombre de points
    switch(res->mode)
    {
        case SPIRALES: // Si le rseau analyse des spirales
            fread(&(res->offset), sizeof(long), 1, fichier); // On charge l'offset
            fread(&(res->phase), sizeof(double), 1, fichier); // On charge la phase
            generer_spirales(res, res->nbr_pts, res->offset, res->coord); // On regnre les points exemples
            break;
        case PERSO: // Dans le cas d'un dessin on fait une lecture par dfaut
        default:
            /// Allocation mmoire pour l'ensemble d'exemple
            res->pts = (double **)malloc(res->nbr_pts * sizeof(double *));
            if(!res->pts) {exit(EXIT_FAILURE);}
            res->att = (double **)malloc(res->nbr_pts * sizeof(double *));
            if(!res->att) {exit(EXIT_FAILURE);}
            for(i = 0; i < res->nbr_pts; i++) // Pour chaque point  apprendre
            {
                res->pts[i] = (double *)malloc(res->nbr_neurones[0] * sizeof(double)); // Il y a autant d'entres que de neurones d'entres
                if(!res->pts[i]) {exit(EXIT_FAILURE);}
                res->att[i] = (double *)malloc(res->nbr_neurones[res->nbr_couches - 1] * sizeof(double)); // Il y a autant de sorties que de neurones de sorties
                if(!res->att[i]) {exit(EXIT_FAILURE);}
            }

            for(i = 0; i < res->nbr_pts; i++) // Pour chaque exemple
            {
                fread(res->pts[i], sizeof(double), t2[0], fichier); // On charge toutes les entres de cet exemple
                fread(res->att[i], sizeof(double), t2[res->nbr_couches - 1], fichier); // Et les sorties attendues
            }
            break;
    }
    fread(res->temps, sizeof(long), 1, fichier); // On charge le temps d'apprentissage

    free(t2); // On nettoie
    fclose(fichier); // On ferme le fichier
    return 1;
}

double calculer_reseau(t_reseau res, double *entree, double *sortie, double *attendue)
{
    long i, j, k; // Compteurs
    double max_delta; // Si apprentissage : plus grande valeur absolue parmis les facteurs de correction d'erreur

    max_delta = 0.0; // Si on ne passe pas par l'apprentissage alors on renverra 0.0 (aucun poids n'est chang !)
    /// Calcul de la sortie correspondant  l'entre avec le rseau de neurones actuel
    res.resultat[0] = entree;
    res.resultat[res.nbr_couches - 1] = sortie;
    for(i = 1; i < res.nbr_couches; i++) // Pour chaque couche (sauf celle d'entre puisque le "calcul" est dj fait)
    {
        for(j = 0; j < res.nbr_neurones[i]; j++) // Pour chaque neurone de cette couche
        {
            res.resultat[i][j] = 0.0; // Initialisation pour la somme
            for(k = 0; k < res.nbr_neurones[i - 1]; k++) {res.resultat[i][j] += res.poids[i][j][k] * res.resultat[i - 1][k];} // Calcul de la somme pondre des entres
            res.resultat[i][j] = tanh(res.resultat[i][j]); /// Calcul de la sortie
        }
    }
    /// Fin du calcul de la sortie

    /// Apprentissage
    if(attendue) // Si on est en phase d'apprentissage
    {
        /// Calcul des facteurs de correction d'erreur
        for(j = 0; j < res.nbr_neurones[res.nbr_couches - 1]; j++) // Cas particulier de la couche de sortie (i = res.nbr_couche - 1)
        {
            res.delta[res.nbr_couches - 1][j] = (1.0 - res.resultat[res.nbr_couches - 1][j] * res.resultat[res.nbr_couches - 1][j]) * (attendue[j] - res.resultat[res.nbr_couches - 1][j]); // Calcul de l'erreur avec la drive de th
        }
        for(i = res.nbr_couches - 2; i > 0; i--) // Pour chaque couche (sauf celle de sortie (cas particulier) et d'entre (pas de poids en entre)) dans l'ordre inverse (RETROpropagation)
        {
            for(j = 0; j < res.nbr_neurones[i]; j++) // Pour chaque neurone de cette couche
            {
                res.delta[i][j] = 0.0; // Initialisation pour la somme
                for(k = 0; k < res.nbr_neurones[i + 1]; k++) {res.delta[i][j] += res.delta[i + 1][k] * res.poids[i + 1][k][j];} // Calcul de la somme pondre par les poids en sortie
                res.delta[i][j] = (1.0 - res.resultat[i][j] * res.resultat[i][j]) * res.delta[i][j]; /// Calcul du facteur de correction avec la drive de th
            }
        }
        /// Fin de calcul des facteurs de correction d'erreur

        /// Mise  jour des poids
        max_delta = fabs(res.delta[1][0]); // Initialisation du maximum pour la recherche
        for(i = 1; i < res.nbr_couches; i++) // Pour chaque couche (sauf celle d'entre puisque il n'y a pas de poids en entres)
        {
            for(j = 0; j < res.nbr_neurones[i]; j++) // Pour chaque neurone de cette couche
            {
                if(max_delta < fabs(res.delta[i][j])) {max_delta = fabs(res.delta[i][j]);} // Actualisation du maximum si ncessaire
                for(k = 0; k < res.nbr_neurones[i - 1]; k++) // Pour chaque entre de ce neurone
                {
                    res.poids[i][j][k] += res.epsilon * res.delta[i][j] * res.resultat[i - 1][k]; /// Correction du poids
                }
            }
        }
        /// Fin de la mise  jour
    }
    /// Fin de l'apprentissage
    return max_delta;
}

long apprentissage(t_reseau res, t_interface *aff)
{
    SDL_Event event;
    double *out;
    double delta, seuil;
    long i, j, t, continuer;
    long temps = SDL_GetTicks(); // Temps rel pour un affichage rgulier
    long mesure; // Mesure du temps de calcul (!= du temps rellement pass car le processeur peut passer du temps sur d'autres applications)

    // Allocation de la mmoire ncessaire
    out = (double *)malloc(res.nbr_neurones[res.nbr_couches - 1] * sizeof(double));
    if(!out)
    {
        free(out);
        return 0;
    }

    t = obtenir_element(aff, CADRE_RESULTAT);
    seuil = res.seuil * aff->options.pts_boucle; // Evite de calculer la moyenne  chaque fois
    mesure = clock();
    continuer = 1;
    do
    {
        delta = 0.0;
        for(i = 0; i < aff->options.pts_boucle; i++) // On va prsenter NBR_PTS_BOUCLE exemples au rseau
        {
            j = rand() % res.nbr_pts; // Choix altoire d'un exemple
            delta += calculer_reseau(res, res.pts[j], out, res.att[j]); // Traitement de l'exemple
        }

        if (SDL_GetTicks() - temps > aff->options.apprentissage_ms) // S'il faut actualiser l'aperu
        {
            afficher_resultat(res, aff->gui[t].image, aff->options.apercu_resolution); // Affichage du rsultat
            //printf("%lf\n", delta);
            SDL_BlitSurface(aff->gui[t].image, NULL, aff->fenetre, &(aff->gui[t].position)); // On colle le rsultat sur la fentre
            SDL_Flip(aff->fenetre); // On actualise l'cran
            temps = SDL_GetTicks();
        }

        j = 1;
        while(SDL_PollEvent(&event)) // On rcupre les venements
        {
            switch(event.type)// En fonction du type d'evenement
            {
                case SDL_QUIT: // Si on clique sur la petite croix (ou Alt+F4, etc...)
                    continuer = 0; // On prcise qu'on veut quitter la boucle
                    aff->continuer = 0; // Et le programme
                    j = 0;
                    break;
                case SDL_KEYDOWN: // Si une touche vient d'tre enfonce
                    switch(event.key.keysym.sym) // On vrifie quelle touche a t enfonce
                    {
                        case SDLK_ESCAPE: // Si c'est la touche Echapper
                            continuer = 0;  // On prcise qu'on veut quitter la boucle
                            j = 0;
                            break;
                        default: // On ne s'occupe pas des autres touches
                            break;
                    }
                default: // On ne traite pas les autres vnements
                    break;
            }
        }
    }while(continuer && delta > seuil); // Tant que le delta moyen n'est pas sous le seuil objectif, on continue l'apprentissage (sauf si on veut quitter...)
    *(res.temps) += (clock() - mesure) * 1000 / CLOCKS_PER_SEC;

    free(out); // Petit nettoyage
    return j;
}

// Fonctionne avec plusieurs entres/sorties
/*long afficher_resultat(t_reseau res, SDL_Surface *cadre, long facteur)
{
    SDL_Rect point; // Position et aire d'un point  dessiner
    SDL_Color couleur; // Couleur du point  dessiner
    double *in, *out; // Entres / Sorties du rseau
    double temp1, temp2, r, g, b;
    long i;

    // Initialisation / Allocation mmoire
    in = (double *)malloc(res.nbr_neurones[0] * sizeof(double));
    if(!in) {return 0;}
    out = (double *)malloc(res.nbr_neurones[res.nbr_couches - 1] * sizeof(double));
    if(!out) {free(in); return 0;}
    point.x = 0;
    point.y = 0;
    point.w = facteur;
    point.h = facteur;

    // Pour chaque point  dessiner
    for(point.y = 0; point.y < cadre->h; point.y += facteur)
    {
        for(point.x = 0; point.x < cadre->w; point.x += facteur)
        {
            // "Centre du point"
            in[0] = point.x + (facteur - 1) / 2;
            in[1] = cadre->h - point.y - (facteur - 1) / 2;
            // Normalisation
            switch(res.coord) // En fonction du systme de coordonnes utilises
            {
                case POL: // Coordonnes polaires
                    in[0] -= cadre->w / 2;
                    in[1] -= cadre->h / 2;
                    cart_pol(in, in);
                    in[0] /= sqrt((cadre->w * cadre->w + cadre->h * cadre->h) / 8);
                    in[1] /= 2.0 * M_PI;
                    break;
                case CART: // On calcul avec les coordonnes cartsiennes par dfaut
                default:
                    in[0] /= cadre->w;
                    in[1] /= cadre->h;
                    break;
            }
            // Calcul de la couleur
            calculer_reseau(res, in, out, NULL);
            temp1 = 0.0; temp2 = 0.0;
            r = 0.0; g = 0.0; b = 0.0;
            for(i = 0; i < res.nbr_neurones[res.nbr_couches - 1]; i++) {temp1 += out[i];}
            for(i = 0; i < res.nbr_neurones[res.nbr_couches - 1]; i++)
            {
                out[i] /= temp1;
                if(out[i] < 0.0) {out[i] = 0.0;}
                else if(out[i] > 1.0) {out[i] = 1.0;}
                temp2 += out[i];
                r += out[i] * res.forme[i].r; g += out[i] * res.forme[i].g; b += out[i] * res.forme[i].b;
            }
            couleur.r = r / temp2;
            couleur.g = g / temp2;
            couleur.b = b / temp2;
            SDL_FillRect(cadre, &point, SDL_MapRGB(cadre->format, couleur.r, couleur.g, couleur.b)); // Affichage
        }
    }

    // Affichage des points exemple
    for(i = 0; i < res.nbr_pts; i++)
    {
        // Placement
        switch(res.coord) // En fonction du systme de coordonnes utilises
        {
            case POL: // Coordonnes polaires
                point.x = cadre->w / 2 * (1.0 + res.pts[i][0] * cos(res.pts[i][1] * 2.0 * M_PI)) - facteur / 2;
                point.y = cadre->h / 2 * (1.0 - res.pts[i][0] * sin(res.pts[i][1] * 2.0 * M_PI)) - facteur / 2;
                break;
            case CART: // On calcul avec les coordonnes cartsiennes
                point.x = cadre->w * res.pts[i][0] - facteur / 2;
                point.y = cadre->h * (1.0 - res.pts[i][1]) - facteur / 2;
                break;
            default:
                break;
        }
        SDL_FillRect(cadre, &point, SDL_MapRGB(cadre->format, 255, 255, 255)); // Affichage
    }

    // Nettoyage et fin
    free(in);
    free(out);
    return 1;
}*/

// Version optimise pour 2 entres et 2 sorties
long afficher_resultat(t_reseau res, SDL_Surface *cadre, long facteur)
{
    SDL_Rect point; // Position et aire d'un point  dessiner
    SDL_Color couleur; // Couleur du point  dessiner
    double in[2], out[2]; // Entres / Sorties du rseau
    double temp1, temp2, temp3;
    long i, temp4;

    // Initialisation
    point.x = 0;
    point.y = 0;
    point.w = facteur;
    point.h = facteur;
    temp3 = sqrt((cadre->w * cadre->w + cadre->h * cadre->h) / 8);
    temp4 = (facteur - 1) / 2;

    // Pour chaque point  dessiner
    for(point.y = 0; point.y < cadre->h; point.y += facteur)
    {
        for(point.x = 0; point.x < cadre->w; point.x += facteur)
        {
            // Normalisation
            switch(res.coord) // En fonction du systme de coordonnes utilises
            {
                case CART: // On calcul avec les coordonnes cartsiennes
                    in[0] = (point.x + temp4) / (double) cadre->w;
                    in[1] = (cadre->h - point.y - temp4) / (double) cadre->h;
                    break;
                case POL: // Coordonnes polaires
                    in[0] = (point.x + temp4) - cadre->w / 2;
                    in[1] = (cadre->h - point.y - temp4) - cadre->h / 2;
                    cart_pol(in, in);
                    in[0] /= temp3;
                    in[1] /= 2.0 * M_PI;
                    break;
                default:
                    break;
            }

            // Calcul de la couleur
            calculer_reseau(res, in, out, NULL);
            temp1 = out[0] / (out[0] + out[1]);
            temp2 = out[1] / (out[0] + out[1]);
            if(temp1 < 0.0) {temp1 = 0.0;}
            else if(temp1 > 1.0) {temp1 = 1.0;}
            if(temp2 < 0.0) {temp2 = 0.0;}
            else if(temp2 > 1.0) {temp2 = 1.0;}
            couleur.r = (temp1 * res.forme[0].r + temp2 * res.forme[1].r) / (temp1 + temp2);
            couleur.g = (temp1 * res.forme[0].g + temp2 * res.forme[1].g) / (temp1 + temp2);
            couleur.b = (temp1 * res.forme[0].b + temp2 * res.forme[1].b) / (temp1 + temp2);
            SDL_FillRect(cadre, &point, SDL_MapRGB(cadre->format, couleur.r, couleur.g, couleur.b)); // Affichage
        }
    }

    // Affichage des points exemple
    for(i = 0; i < res.nbr_pts; i++)
    {
        // Placement
        switch(res.coord) // En fonction du systme de coordonnes utilises
        {
            case CART: // On calcul avec les coordonnes cartsiennes
                point.x = cadre->w * res.pts[i][0] - temp4;
                point.y = cadre->h * (1.0 - res.pts[i][1]) - temp4;
                break;
            case POL: // Coordonnes polaires
                point.x = cadre->w / 2 * (1.0 + res.pts[i][0] * cos(res.pts[i][1] * 2.0 * M_PI)) - temp4;
                point.y = cadre->h / 2 * (1.0 - res.pts[i][0] * sin(res.pts[i][1] * 2.0 * M_PI)) - temp4;
                break;
            default:
                break;
        }
        SDL_FillRect(cadre, &point, SDL_MapRGB(cadre->format, 255, 255, 255)); // Affichage (en blanc)
	point.w = facteur; // Malheureusement SDL modifie parfois ces donnes... donc on les remet
	point.h = facteur;
    }
    return 1;
}
